/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright 
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package gov.va.med.term.sopt.mojo;

import java.io.File;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;

import gov.va.med.term.sopt.data.EnumValidatedTableData;
import gov.va.med.term.sopt.data.EnumValidatedTableDataReader;
import gov.va.med.term.sopt.propertyTypes.PT_Annotations;
import gov.va.med.term.sopt.propertyTypes.PT_Descriptions;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Refsets;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;

/**
 * {@link SOPTImportMojo}
 * 
 * Goal which converts SOPT data into the workbench jbin format
 * 
 * @author <a href="mailto:nmarques@westcoastinformatics.com">Nuno Marques</a>
 */
@Mojo(name = "convert-SOPT-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES)
public class SOPTImportMojo extends ConverterBaseMojo
{
	private IBDFCreationUtility importUtil_;

	private HashMap<UUID, String> loadedConcepts = new HashMap<>();

	private PropertyType attributes_;
	private PT_Descriptions descriptions_;
	private BPT_Refsets refsets_;

	private Map<String, UUID> parentConcepts = new HashMap<>();

	private final static String REFSET_NAME = "SOPT";
	private final static String REFSET_PROPERTY_NAME = "All SOPT Concepts";
	private final static String COMPONENT_REFERENCE_METADATA = "SOPT Metadata";

	private UUID allSoptConceptsRefset;

	private int conceptCount = 0;

	@Override
	public void execute() throws MojoExecutionException {
		try {
			super.execute();

			Date date = new Date();

			importUtil_ = new IBDFCreationUtility(Optional.of("SOPT " + converterSourceArtifactVersion),
					Optional.of(MetaData.SOPT_MODULES), outputDirectory, converterOutputArtifactId,
					converterOutputArtifactVersion, converterOutputArtifactClassifier, false, date.getTime());

			attributes_ = new PT_Annotations();
			descriptions_ = new PT_Descriptions();

			refsets_ = new BPT_Refsets(REFSET_NAME);
			refsets_.addProperty(REFSET_PROPERTY_NAME);

			// Every time concept created, add membership to "All SOPT Concepts"
			allSoptConceptsRefset = refsets_.getProperty(REFSET_PROPERTY_NAME).getUUID();

			// Switch on version to select proper Columns enum to use in constructing reader
			final EnumValidatedTableDataReader<ColumnsV1> importer = new EnumValidatedTableDataReader<>(
					inputFileLocation, ColumnsV1.class);
			final EnumValidatedTableData<ColumnsV1> terminology = importer.process();

			ConsoleUtil.println("Loaded Terminology containing " + terminology.rows().size() + " entries");

			// COLUMNS from ColumnsV1:
			// Concept Code - Hierarchical numbering eg. 1, 11, 111, 112, 113, 12, 121, 122, 129, 13, 14, 2, 21
			// Concept Name - name of concept
			// Preferred Concept Name - all rows the same as Concept Name except 1.
			// Preferred Alternate Code - all are null
			// Code System OID - 2.16.840.1.113883.3.221.5
			// Code System Name - Source of Payment Typology (PHDSC)
			// Code System Code - PH_SourceOfPaymentTypology_PHDSC
			// Code System Version - 7.0
			// HL7 Table 0396 Code - PHDSCPT

			// Parent soptMetadata ComponentReference
			final ComponentReference soptMetadata = ComponentReference
					.fromConcept(createType(MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid(),
							COMPONENT_REFERENCE_METADATA + IBDFCreationUtility.metadataSemanticTag_));

			// loadTerminologyMetadataAttributes onto soptMetadata
			importUtil_.loadTerminologyMetadataAttributes(converterSourceArtifactVersion,
					Optional.empty(), converterOutputArtifactVersion,
					Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);

			// load metadata
			importUtil_.loadMetaDataItems(Arrays.asList(attributes_, refsets_, descriptions_),
					soptMetadata.getPrimordialUuid());

			// Create SOPT root concept under ISAAC_ROOT
			final ConceptChronology<? extends ConceptVersion<?>> soptRootConcept = importUtil_
					.createConcept(REFSET_NAME, true, MetaData.ISAAC_ROOT.getPrimordialUuid());
			ConsoleUtil
					.println("Created SOPT root concept " + soptRootConcept.getPrimordialUuid() + " under ISAAC_ROOT");

//			// Code System OID
//			final Map<String, ConceptChronology<? extends ConceptVersion<?>>> codeSystemOIDValueConceptByValueMap = new HashMap<>();
//			ConsoleUtil.println("attributes_ is null = " + (attributes_ == null));
//			final UUID codeSystemOIDUuid = attributes_.getProperty(ColumnsV1.CodeSystemOID.toString()).getUUID();
//
//			for (String value : terminology.getDistinctValues(ColumnsV1.CodeSystemOID)) {
//				if (StringUtils.isBlank(value)) {
//					throw new RuntimeException("Cannot load SOPT data with blank Code System OID");
//				}
//
//				// Create the Concept System OID concept as a child of both SOPT root and the Concept Name property
//				// metadata concept and store in map for later retrieval
//				final UUID valueConceptUuid = ConverterUUID
//						.createNamespaceUUIDFromString(codeSystemOIDUuid.toString() + "|" + value, true);
//				final ConceptChronology<? extends ConceptVersion<?>> valueConcept = importUtil_.createConcept(
//						valueConceptUuid, value, null, null, null, codeSystemOIDUuid,
//						soptRootConcept.getPrimordialUuid());
//
//				ConsoleUtil.println("Created SOPT Code value concept " + valueConcept.getPrimordialUuid() + " for \""
//						+ value + "\" under parents Code System OID property " + codeSystemOIDUuid
//						+ " and SOPT root concept " + soptRootConcept);
//
//				// Store Code System OID value concept in map by value
//				codeSystemOIDValueConceptByValueMap.put(value, valueConcept);
//			}
//
//			// Code System Name
//			final Map<String, ConceptChronology<? extends ConceptVersion<?>>> codeSystemNameValueConceptByValueMap = new HashMap<>();
//			final UUID codeSystemNameUuid = attributes_.getProperty(ColumnsV1.CodeSystemName.toString()).getUUID();
//
//			for (String value : terminology.getDistinctValues(ColumnsV1.CodeSystemName)) {
//				if (StringUtils.isBlank(value)) {
//					throw new RuntimeException("Cannot load SOPT data with blank Code System Name");
//				}
//
//				// Create the Concept System Name concept as a child of both SOPT root and the Concept Name property
//				// metadata concept and store in map for later retrieval
//				final UUID valueConceptUuid = ConverterUUID
//						.createNamespaceUUIDFromString(codeSystemNameUuid.toString() + "|" + value, true);
//				final ConceptChronology<? extends ConceptVersion<?>> valueConcept = importUtil_.createConcept(
//						valueConceptUuid, value, null, null, null, codeSystemNameUuid,
//						soptRootConcept.getPrimordialUuid());
//
//				ConsoleUtil.println("Created SOPT Code  value concept " + valueConcept.getPrimordialUuid() + " for \""
//						+ value + "\" under parents Code System Name property " + codeSystemNameUuid
//						+ " and SOPT root concept " + soptRootConcept);
//
//				// Store Code System Name value concept in map by value
//				codeSystemNameValueConceptByValueMap.put(value, valueConcept);
//			}
//
//			// Code System Code
//			final Map<String, ConceptChronology<? extends ConceptVersion<?>>> codeSystemCodeValueConceptByValueMap = new HashMap<>();
//			final UUID codeSystemCodeUuid = attributes_.getProperty(ColumnsV1.CodeSystemCode.toString()).getUUID();
//
//			for (String value : terminology.getDistinctValues(ColumnsV1.CodeSystemCode)) {
//				if (StringUtils.isBlank(value)) {
//					throw new RuntimeException("Cannot load SOPT data with blank Code System Code");
//				}
//
//				// Create the Concept System Code concept as a child of both SOPT root and the Concept Name property
//				// metadata concept and store in map for later retrieval
//				final UUID valueConceptUuid = ConverterUUID
//						.createNamespaceUUIDFromString(codeSystemCodeUuid.toString() + "|" + value, true);
//				final ConceptChronology<? extends ConceptVersion<?>> valueConcept = importUtil_.createConcept(
//						valueConceptUuid, value, null, null, null, codeSystemCodeUuid,
//						soptRootConcept.getPrimordialUuid());
//
//				ConsoleUtil.println("Created SOPT Code  value concept " + valueConcept.getPrimordialUuid() + " for \""
//						+ value + "\" under parents Code System Code property " + codeSystemCodeUuid
//						+ " and SOPT root concept " + soptRootConcept);
//
//				// Store Code System Code value concept in map by value
//				codeSystemCodeValueConceptByValueMap.put(value, valueConcept);
//			}
//
//			// Code System Version
//			final Map<String, ConceptChronology<? extends ConceptVersion<?>>> codeSystemVersionValueConceptByValueMap = new HashMap<>();
//			final UUID codeSystemVersionUuid = attributes_.getProperty(ColumnsV1.CodeSystemVersion.toString())
//					.getUUID();
//
//			for (String value : terminology.getDistinctValues(ColumnsV1.CodeSystemVersion)) {
//				if (StringUtils.isBlank(value)) {
//					throw new RuntimeException("Cannot load SOPT data with blank Code Concept Version");
//				}
//
//				// Create the Concept System Code concept as a child of both SOPT root and the Concept Name property
//				// metadata concept and store in map for later retrieval
//				final UUID valueConceptUuid = ConverterUUID
//						.createNamespaceUUIDFromString(codeSystemCodeUuid.toString() + "|" + value, true);
//				final ConceptChronology<? extends ConceptVersion<?>> valueConcept = importUtil_.createConcept(
//						valueConceptUuid, value, null, null, null, codeSystemVersionUuid,
//						soptRootConcept.getPrimordialUuid());
//
//				ConsoleUtil.println("Created SOPT Code  value concept " + valueConcept.getPrimordialUuid() + " for \""
//						+ value + "\" under parents Code System Version property " + codeSystemVersionUuid
//						+ " and SOPT root concept " + soptRootConcept);
//
//				// Store Code System Version value concept in map by value
//				codeSystemVersionValueConceptByValueMap.put(value, valueConcept);
//			}
//
//			// HL7 Table 0396 Code
//			final Map<String, ConceptChronology<? extends ConceptVersion<?>>> hl7Table3096CodeValueConceptByValueMap = new HashMap<>();
//			final UUID hl7Table3096CodeUuid = attributes_.getProperty(ColumnsV1.HL7Table0396Code.toString()).getUUID();
//
//			for (String value : terminology.getDistinctValues(ColumnsV1.HL7Table0396Code)) {
//				if (StringUtils.isBlank(value)) {
//					throw new RuntimeException("Cannot load SOPT data with blank HL7 Table 0396 Code");
//				}
//
//				// Create the Concept System Code concept as a child of both SOPT root and the Concept Name property
//				// metadata concept and store in map for later retrieval
//				final UUID valueConceptUuid = ConverterUUID
//						.createNamespaceUUIDFromString(codeSystemCodeUuid.toString() + "|" + value, true);
//				final ConceptChronology<? extends ConceptVersion<?>> valueConcept = importUtil_.createConcept(
//						valueConceptUuid, value, null, null, null, codeSystemVersionUuid,
//						soptRootConcept.getPrimordialUuid());
//
//				ConsoleUtil.println("Created SOPT Code  value concept " + valueConcept.getPrimordialUuid() + " for \""
//						+ value + "\" under parents HL7 Table 3096 property " + hl7Table3096CodeUuid
//						+ " and SOPT root concept " + soptRootConcept);
//
//				// Store HL7 Table 0396 Code value concept in map by value
//				hl7Table3096CodeValueConceptByValueMap.put(value, valueConcept);
//			}

			// Populate hierarchy, one row at a time
			for (Map<ColumnsV1, String> row : terminology.rows()) {

				// if code is not a number, it is the header row, skip the row
				if (!StringUtils.isNumeric(row.get(ColumnsV1.ConceptCode))) {

//					final ConceptChronology<? extends ConceptVersion<?>> codeSystemOIDConcept = row
//							.get(ColumnsV1.CodeSystemName) != null
//									? codeSystemOIDValueConceptByValueMap.get(row.get(ColumnsV1.CodeSystemOID)) : null;
//
//					final ConceptChronology<? extends ConceptVersion<?>> codeSystemNameConcept = row
//							.get(ColumnsV1.CodeSystemName) != null
//									? codeSystemNameValueConceptByValueMap.get(row.get(ColumnsV1.CodeSystemName))
//									: null;
//
//					final ConceptChronology<? extends ConceptVersion<?>> codeSystemCodeConcept = row
//							.get(ColumnsV1.CodeSystemVersion) != null
//									? codeSystemCodeValueConceptByValueMap.get(row.get(ColumnsV1.CodeSystemCode))
//									: null;
//
//					final ConceptChronology<? extends ConceptVersion<?>> codeSystemVersionConcept = row
//							.get(ColumnsV1.CodeSystemVersion) != null
//									? codeSystemVersionValueConceptByValueMap.get(row.get(ColumnsV1.CodeSystemVersion))
//									: null;
//
//					final ConceptChronology<? extends ConceptVersion<?>> hl7Table0396CodeConcept = row
//							.get(ColumnsV1.HL7Table0396Code) != null
//									? hl7Table3096CodeValueConceptByValueMap.get(row.get(ColumnsV1.HL7Table0396Code))
//									: null;
//					
//					//(ComponentReference referencedComponent, UUID uuidForCreatedAnnotation, 
//					//DynamicSememeData value, UUID refexDynamicTypeUuid, State state, Long time)
//									
//					// add annotations Code System OID NID
//					importUtil_.addAnnotation(ComponentReference.fromChronology(codeSystemOIDConcept), soptMetadata.getPrimordialUuid(),
//							new DynamicSememeNidImpl(codeSystemOIDConcept.getNid()), codeSystemOIDUuid, State.ACTIVE,
//							(Long) null);
//
//					// add annotations Code System Name NID
//					importUtil_.addAnnotation(ComponentReference.fromChronology(codeSystemNameConcept), soptMetadata.getPrimordialUuid(),
//							new DynamicSememeNidImpl(codeSystemNameConcept.getNid()), codeSystemNameUuid, State.ACTIVE,
//							(Long) null);
//
//					// add annotations Code System Code NID
//					importUtil_.addAnnotation(ComponentReference.fromChronology(codeSystemVersionConcept), soptMetadata.getPrimordialUuid(),
//							new DynamicSememeNidImpl(codeSystemOIDConcept.getNid()), codeSystemCodeUuid, State.ACTIVE,
//							(Long) null);
//
//					// add annotations Code System Version NID
//					importUtil_.addAnnotation(ComponentReference.fromChronology(codeSystemVersionConcept), soptMetadata.getPrimordialUuid(),
//							new DynamicSememeNidImpl(codeSystemVersionConcept.getNid()), codeSystemVersionUuid,
//							State.ACTIVE, (Long) null);
//
//					// add annotations Code System HL7 Table 0396 code NID
//					importUtil_.addAnnotation(ComponentReference.fromChronology(hl7Table0396CodeConcept), soptMetadata.getPrimordialUuid(),
//							new DynamicSememeNidImpl(hl7Table0396CodeConcept.getNid()), hl7Table3096CodeUuid,
//							State.ACTIVE, (Long) null);
					continue;
				}

				// 1, 11, 111, 112, 113, 12, 121, 122, 123, 129, 13, 14, 2, 21
				// ...
				String conceptCode;
				String conceptName;
				String preferredConceptCode;
				String preferredConceptName;

				try {

					conceptCode = row.get(ColumnsV1.ConceptCode);
					conceptName = row.get(ColumnsV1.ConceptName);
					preferredConceptCode = row.get(ColumnsV1.PreferredAlternateCode);
					preferredConceptName = row.get(ColumnsV1.PreferredConceptName);

					UUID parentUuid = findParentUuid(conceptCode);

					ConsoleUtil.println("Creating " + conceptCode + " - " + conceptName);

					// String name, boolean skipDupeCheck
					UUID rowConceptUuid = ConverterUUID.createNamespaceUUIDFromString(conceptCode + "|" + conceptName,
							true);

					// add to map
					parentConcepts.put(conceptCode, rowConceptUuid);
					final ConceptChronology<? extends ConceptVersion<?>> rowConcept;

					rowConcept = importUtil_.createConcept(rowConceptUuid, // UUID conceptPrimordialUuid,
							conceptName, // String fsn,
							true, // boolean createSynonymFromFSN,
							(parentUuid == null) ? soptRootConcept.getPrimordialUuid() : parentUuid); // relParentPrimordial);

					final ComponentReference rowComponentReference = ComponentReference.fromConcept(rowConcept);

					importUtil_.addDescription(rowComponentReference, conceptCode, DescriptionType.FSN, true, null,
							State.ACTIVE);

					if (!conceptName.equals(preferredConceptName)) {
						importUtil_.addDescription(rowComponentReference, preferredConceptName, DescriptionType.SYNONYM,
								true, null, State.ACTIVE);
					}

					if (preferredConceptCode != null && StringUtils.isNotEmpty(preferredConceptCode)) {
						addDescription(rowComponentReference, preferredConceptCode, DescriptionType.SYNONYM,
								descriptions_.getProperty("PreferredConceptCode").getUUID(), false);
					}

					importUtil_.addStaticStringAnnotation(rowComponentReference, conceptCode,
							MetaData.CODE.getPrimordialUuid(), State.ACTIVE);

					// add refset
					importUtil_.addRefsetMembership(rowComponentReference, allSoptConceptsRefset, State.ACTIVE,
							(Long) null);

					++conceptCount;

				} catch (Exception e) {
					final String msg = "Failed processing row with " + e.getClass().getSimpleName() + " "
							+ e.getLocalizedMessage() + ": " + row;
					ConsoleUtil.println(msg);
					throw new RuntimeException(msg, e);
				}
			}

			ConsoleUtil.println("Metadata load stats");
			for (String line : importUtil_.getLoadStats().getSummary()) {
				ConsoleUtil.println(line);
			}
			importUtil_.clearLoadStats();

			ConsoleUtil.println("Processed " + conceptCount + " concepts");

			ConsoleUtil.println("Load Statistics");

			// this could be removed from final release. Just added to help debug editor problems.
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, "soptUuid");

			importUtil_.shutdown();
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
		} catch (Exception ex) {
			throw new MojoExecutionException(ex.getLocalizedMessage(), ex);
		}
	}

	// find UUID of parent record. eg. the parent of record 111 is 11.
	// there may be an instance where the parent of a record is not the
	// same key less 1 character. eg. 9999's parent record is 99
	private UUID findParentUuid(String conceptCode) {

		UUID parentUUID = null;
		if (conceptCode != null && conceptCode.length() > 1) {
			for (int i = conceptCode.length() - 1; i > 0; i--) {
				parentUUID = parentConcepts.get(conceptCode.substring(0, i));
				if (parentUUID != null) {
					break;
				}
			}
		}
		return parentUUID;
	}

	private void addDescription(ComponentReference concept, String text, DescriptionType descriptionType,
			UUID extendedType, boolean preferred) {
		UUID descriptionPrimordialUUID = ConverterUUID.createNamespaceUUIDFromStrings(
				concept.getPrimordialUuid().toString(), text, extendedType.toString(), descriptionType.name(),
				new Boolean(preferred).toString());
		importUtil_.addDescription(concept, descriptionPrimordialUUID, text, descriptionType, preferred, extendedType,
				State.ACTIVE);
	}

	private ConceptChronology<? extends ConceptVersion<?>> createType(UUID parentUuid, String typeName)
			throws Exception {
		ConceptChronology<? extends ConceptVersion<?>> concept = importUtil_.createConcept(typeName, true);
		loadedConcepts.put(concept.getPrimordialUuid(), typeName);
		importUtil_.addParent(ComponentReference.fromConcept(concept), parentUuid);
		return concept;
	}

	public static void main(String[] args) throws MojoExecutionException {
		SOPTImportMojo i = new SOPTImportMojo();
		i.outputDirectory = new File("../sopt-ibdf/target");
		i.inputFileLocation = new File("../sopt-ibdf/target/generated-resources/src/");
		i.converterOutputArtifactVersion = "2016.06";
		i.converterVersion = "SNAPSHOT";
		i.converterSourceArtifactVersion = "7.0";
		i.execute();
	}
}